program HEATEQN;
{--------------------------------------------------------------------}
{  Alg10'23.pas   Pascal program for implementing Algorithm 10.2-3   }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 10.2 (Forward-Difference Method for the Heat Equation). }
{  Section   10.2, Parabolic Equations, Page 516                     }
{                                                                    }
{  Algorithm 10.3 (Crank-Nicholson Method for the Heat Equation).    }
{  Section   10.2, Parabolic Equations, Page 517                     }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    Pi = 3.1415926535;
    GNmax = 630;
    MaxN = 26;
    MaxM = 101;
    FunMax = 9;

  type
    VECTOR = array[1..MaxN] of real;
    MATRIX = array[1..MaxN, 1..MaxM] of real;
    LETTER = string[8];
    States = (Changes, Done, Working);
    DoSome = (Go, Stop);
    LETTERS = string[200];

  var
    FunType, GNpts, Inum, M, Mend, Meth, N, Order, Sub: integer;
    A, B, C, C1, C2, Rnum, Y0: real;
    Ans: CHAR;
    U: MATRIX;
    State: States;
    DoMo: DoSome;
    Mess: LETTERS;

  function F (X: real): real;
  begin
    case FunType of
      1: 
        F := SIN(Pi * X) + SIN(3 * Pi * X);
    end;
  end;

  function G1 (T: real): real;
  begin
    case FunType of
      1: 
        G1 := 0;
    end;
  end;

  function G2 (T: real): real;
  begin
    case FunType of
      1: 
        G2 := 0;
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer);
  begin
    case FunType of
      1: 
        begin
          WRITELN;
          WRITELN('          The boundary functions are:');
          WRITELN;
          WRITELN('          u(x,0) =  f(x) =  SIN(Pi*X)+SIN(3*Pi*X)');
          WRITELN;
          WRITELN('          u(0,t) = g1(t) = 0');
          WRITELN;
          WRITELN('          u(a,t) = g2(t) = 0');
        end;
    end;
  end;

  procedure ForwDiff ({FUNCTION F(x,t:real), G1(t:real), G2(t:real): real;}
                  A, B, C: real; N, M: integer; var U: MATRIX);
    var
      I, J: integer;
      H, K, R, S: real;

    function Fi (I: integer): real;
    begin
      Fi := F(H * (I - 1));
    end;

    function G1i (I: integer): real;
    begin
      G1i := G1(K * (I - 1));
    end;

    function G2i (I: integer): real;
    begin
      G2i := G2(K * (I - 1));
    end;

  begin                                       {The main program ForwDiff}
    H := A / (N - 1);
    K := B / (M - 1);
    R := C * C * K / H / H;
    S := 1 - 2 * R;
    for J := 1 to M do
      begin
        U[1, J] := G1i(J);
        U[N, J] := G2i(J);
      end;
    for I := 2 to N - 1 do
      begin
        U[I, 1] := Fi(I);
      end;
    for J := 2 to M do
      for I := 2 to N - 1 do
        U[I, J] := S * U[I, J - 1] + R * (U[I - 1, J - 1] + U[I + 1, J - 1]);
  end;

  procedure CrankNich ({FUNCTION F(x,t:real), G1(x,t:real), G2(x,t:real): real;}
                        A, B, C: real; N, M: integer; var U: MATRIX);
    var
      I, J: integer;
      H, K, R, S1, S2: real;
      X, X0, Va, Vb, Vc, Vd: VECTOR;

    function Fi (I: integer): real;
    begin
      Fi := F(H * (I - 1));
    end;

    function G1i (I: integer): real;
    begin
      G1i := G1(K * (I - 1));
    end;

    function G2i (I: integer): real;
    begin
      G2i := G2(K * (I - 1));
    end;

    procedure TriSystem (Va, Vb, Vc, Vd: VECTOR; var X0: VECTOR; N: integer);
      var
        K: integer;
        T: real;
        A0, B0, C0, D0: VECTOR;
    begin
      for K := 1 to N do
        begin
          A0[K] := Va[K];
          B0[K] := Vb[K];
          C0[K] := Vc[K];
          D0[K] := Vd[K];
        end;
      for K := 2 to N do
        begin
          T := A0[K - 1] / D0[K - 1];
          D0[K] := D0[K] - T * C0[K - 1];
          B0[K] := B0[K] - T * B0[K - 1];
        end;
      X0[N] := B0[N] / D0[N];
      for K := N - 1 downto 1 do
        X0[K] := (B0[K] - C0[K] * X0[K + 1]) / D0[K];
    end;

  begin                                       {The main program CrankNich}
    H := A / (N - 1);
    K := B / (M - 1);
    R := C * C * K / H / H;
    S1 := 2 + 2 / R;
    S2 := 2 / R - 2;
    for J := 1 to M do
      begin
        U[1, J] := G1i(J);
        U[N, J] := G2i(J);
      end;
    for I := 2 to N - 1 do
      begin
        U[I, 1] := Fi(I);
      end;
    for I := 1 to N do
      Vd[I] := S1;
    Vd[1] := 1;
    Vd[N] := 1;
    for I := 1 to N - 1 do
      begin
        Va[I] := -1;
        Vc[I] := -1;
      end;
    Va[N - 1] := 0;
    Vc[1] := 0;
    Vb[1] := G1i(1);
    Vb[N] := G2i(1);
    for J := 2 to M do
      begin
        for I := 2 to N - 1 do
          Vb[I] := U[I - 1, J - 1] + U[I + 1, J - 1] + S2 * U[I, J - 1];
        TriSystem(Va, Vb, Vc, Vd, X, N);
        for I := 1 to N do
          U[I, J] := X[I];
      end;
  end;

  procedure MESSAGE (var Meth: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN('                      SOLUTION OF PARABOLIC EQUATIONS');
    WRITELN;
    Meth := 1;
  end;

  procedure INPUT (var FunType, Meth: integer);
    var
      K: integer;
      Ans: LETTER;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('          Solution of the heat equation');
    WRITELN('                                   2          ');
    WRITELN('                     u (x,t)   =  c  u  (x,t)');
    WRITELN('                      t               xx');
    WRITELN;
    WRITELN('          and   u(x,0)  =  f(x)   for  0 < x < A.');
    WRITELN('          and   u(0,t) = g1(t)  and  u(a,t) = g2(t)   for 0<=t<=B.');
    FunType := 1;
    WRITELN;
    WRITELN('          A numerical approximation is computed over the rectangle');
    WRITELN('                             0<=x<=A.');
    WRITELN('                             0<=t<=B.');
    WRITELN('          You must supply the endpoints for the intervals.');
    WRITELN;
    WRITELN('              Choose the method of computation:');
    WRITELN;
    WRITELN('              < 1 > The forward difference method.');
    WRITELN;
    WRITELN('              < 2 > The Crank-Nicholson method.');
    WRITELN;
    Mess := '                    SELECT your method  < 1 or 2 > ?  ';
    WRITE(Mess);
    READLN(Meth);
    if Meth < 1 then
      Meth := 1;
    if Meth > 2 then
      Meth := 2;
    WRITELN;
  end;

  procedure EPOINTS (var A, B, C: real; var N, M: integer; var State: STATES);
    type
      STATUS = (Change, Enter, Done);
      LETTER = string[1];
    var
      I: integer;
      Valu: real;
      Resp: CHAR;
      Stat: STATUS;
  begin
    Stat := Enter;
    if State = Changes then
      Stat := Change;
    while (Stat = Enter) or (Stat = Change) do
      begin
        CLRSCR;
        WRITELN;
        WRITE('                 ');
        PRINTFUNCTION(FunType);
        WRITELN;
        WRITELN;
        if (Stat = Enter) then
          begin
            Mess := '     For the interval [0,A], ENTER the endpoint   A = ';
            WRITE(Mess);
            READLN(A);
            WRITELN;
            Mess := '     For the interval [0,B], ENTER the endpoint   B = ';
            WRITE(Mess);
            READLN(B);
            WRITELN;
            Mess := '                             ENTER the constant   C = ';
            WRITE(Mess);
            READLN(C);
            WRITELN;
            Mess := '                      ENTER the number of steps   N = ';
            WRITE(Mess);
            READLN(N);
            if N < 2 then
              N := 2;
            if N > 25 then
              N := 25;
            WRITELN;
            Mess := '                      ENTER the number of steps   M = ';
            WRITE(Mess);
            READLN(M);
            if M < 2 then
              M := 2;
            if M > 100 then
              M := 100;
          end
        else
          begin
            WRITELN('     For the interval [0,A], the endpoint  is     A =', A : 8 : 4);
            WRITELN;
            WRITELN;
            WRITELN('     For the interval [0,B], the endpoint  is     B =', B : 8 : 4);
            WRITELN;
            WRITELN;
            WRITELN('                              The constant is     C =', C : 8 : 4);
            WRITELN;
            WRITELN;
            WRITELN('                      The number of steps  is     N =  ', N : 2);
            WRITELN;
            WRITELN;
            WRITELN('                      The number of steps  is     M =  ', M : 2);
          end;
        WRITELN;
        WRITELN;
        WRITE('                      Want to make a change ?    <Y/N>  ');
        READLN(Resp);
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Change;
            CLRSCR;
            WRITELN;
            WRITE('                 ');
            PRINTFUNCTION(FunType);
            WRITELN;
            WRITELN('     [0,A] the current endpoint is A =', A : 8 : 4);
            Mess := '     ENTER  the NEW left  endpoint A =  ';
            WRITE(Mess);
            READLN(A);
            WRITELN;
            WRITELN('     [0,B] the current endpoint is B =', B : 8 : 4);
            Mess := '     ENTER  the NEW right endpoint B =  ';
            WRITE(Mess);
            READLN(B);
            WRITELN;
            WRITELN('        The   current constant is  C =', C : 8 : 4);
            Mess := '     Now  ENTER the NEW  constant  C =  ';
            WRITE(Mess);
            READLN(C);
            WRITELN;
            WRITELN('     The  current value of  N  is  N =  ', N : 1);
            Mess := '     Now  ENTER  the NEW value of  N =  ';
            WRITE(Mess);
            READLN(N);
            if (N < 2) then
              N := 2;
            if (N > 25) then
              N := 25;
            WRITELN;
            WRITELN('     The  current value of  M  is  M =  ', M : 1);
            Mess := '     Now  ENTER  the NEW value of  M =  ';
            WRITE(Mess);
            READLN(M);
            if (M < 2) then
              M := 2;
            if (M > 100) then
              M := 100;
          end
        else
          Stat := Done;
      end;
  end;

  procedure RESULTS (FunType: integer; U: MATRIX; N, M: integer);
    var
      I, J: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITE('      ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN;
    WRITELN('          u(x ,t )   .....    u(x   ,t )');
    WRITELN('             2  j                N-1  j');
    WRITELN('--------------------------------------------------------------------------------');
    WRITELN;
    for J := 1 to M do
      begin
        for I := 2 to N - 1 do
          WRITE(U[I, J] : 10 : 6);
        WRITELN;
        if J mod 21 = 0 then
          begin
            WRITELN;
            WRITE('                  Press the <ENTER> key.  ');
            READLN(Ans);
            WRITELN;
            WRITELN;
          end;
      end;
    WRITELN;
    WRITE('                  Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
  end;

begin                                            {Begin Main Program}
  Meth := 1;
  FunType := 1;
  A := 0;
  B := 1;
  Y0 := 0;
  M := 1;
  State := Working;
  while Meth <> 0 do
    begin
      MESSAGE(Meth);
      DoMo := Go;
      while DoMo = Go do
        begin
          INPUT(FunType, Meth);
          while (State = Working) or (State = Changes) do
            begin
              EPOINTS(A, B, C, N, M, State);
              case Meth of
                1: 
                  ForwDiff(A, B, C, N, M, U);
                2: 
                  CrankNich(A, B, C, N, M, U);
              end;
              RESULTS(FunType, U, N, M);
              WRITELN;
              WRITELN;
              WRITE('     Want to use a  different  initial condition ?  <Y/N>  ');
              READLN(Ans);
              WRITELN;
              if (Ans <> 'Y') and (Ans <> 'y') then
                State := Done
              else
                State := Changes;
            end;
          WRITELN;
          WRITE('     Want to  change  the  differential equation ?  <Y/N>  ');
          READLN(Ans);
          if (Ans <> 'Y') and (Ans <> 'y') then
            DoMo := Stop
          else
            State := Changes;
        end;
      Mess := 'Want to try another method of approximation ?  <Y/N>  ';
      Ans := 'N';
      if (Ans <> 'Y') and (Ans <> 'y') then
        Meth := 0
      else
        State := Changes;
    end;
end.                                            {End of Main Program}

